'use strict';

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/* --------------------------------------------------------------
 cart_handler.js 2021-10-19
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2021 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Component for handling the add to cart and wishlist features
 * at the product details and the category listing pages. It cares
 * for attributes, properties, quantity and all other
 * relevant data for adding an item to the basket or wishlist
 */
gambio.widgets.module('cart_handler', ['hooks', 'form', 'xhr', 'loading_spinner', gambio.source + '/libs/events', gambio.source + '/libs/modal.ext-magnific', gambio.source + '/libs/modal'], function (data) {

	'use strict';

	// ########## VARIABLE INITIALIZATION ##########

	var $this = $(this),
	    $body = $('body'),
	    $window = $(window),
	    busy = false,
	    ajax = null,
	    timeout = 0,
	    defaults = {
		// AJAX "add to cart" URL
		addCartUrl: 'shop.php?do=Cart/BuyProduct',
		// AJAX "add to cart" URL for customizer products
		addCartCustomizerUrl: 'shop.php?do=Cart/Add',
		// AJAX URL to perform a value check
		checkUrl: 'shop.php?do=CheckStatus',
		// AJAX URL to perform the add to wishlist
		wishlistUrl: 'shop.php?do=WishList/Add',
		// Submit URL for price offer button
		priceOfferUrl: 'gm_price_offer.php',
		// Submit method for price offer
		priceOfferMethod: 'get',
		// Selector for the cart dropdown
		dropdown: '#head_shopping_cart',
		// "Add to cart" buttons selectors
		cartButtons: '.js-btn-add-to-cart',
		// "Wishlist" buttons selectors
		wishlistButtons: '.btn-wishlist',
		// "Price offer" buttons selectors
		priceOfferButtons: '.btn-price-offer',
		// Selector for the attribute fields
		attributes: '.js-calculate',
		// Selector for product property
		productOptions: '.modifier-group .modifier-content .modifier-item',
		productOptionField: '.hidden-input',
		// Selector for the quantity
		quantity: '.js-calculate-qty',
		// URL where to get the theme for the dropdown
		tpl: null,
		// Show attribute images in product images swiper (if possible)
		// -- this feature is not supported yet --
		attributImagesSwiper: false,
		// Trigger the attribute images to this selectors
		triggerAttrImagesTo: '#product_image_swiper, #product_thumbnail_swiper, ' + '#product_thumbnail_swiper_mobile',
		// Class that gets added to the button on processing
		processingClass: 'loading',
		// Duration for that the success or fail class gets added to the button
		processingDuration: 2000,
		// AJAX response content selectors
		selectorMapping: {
			buttons: '.shopping-cart-button',
			giftContent: '.gift-cart-content-wrapper',
			giftLayer: '.gift-cart-layer',
			shareContent: '.share-cart-content-wrapper',
			shareLayer: '.share-cart-layer',
			hiddenOptions: '#cart_quantity .hidden-options',
			message: '.global-error-messages',
			messageCart: '.cart-error-msg',
			messageHelp: '.help-block',
			modelNumber: '.model-number',
			modelNumberText: '.model-number-text',
			price: '.current-price-container',
			modifiersForm: '.modifiers-selection',
			quantity: '.products-quantity-value',
			quantityInfo: '.products-quantity',
			ribbonSpecial: '.ribbon-special',
			shippingInformation: '#shipping-information-layer',
			shippingTime: '.products-shipping-time-value',
			shippingTimeImage: '.img-shipping-time img',
			totals: '#cart_quantity .total-box',
			weight: '.products-details-weight-container span',
			abroadShippingInfo: '.abroad-shipping-info'
		},
		page: 'product-listing'
	},
	    options = $.extend(true, {}, defaults, data),
	    module = {},
	    mobile = $(window).width() <= 767;

	// ########## HELPER FUNCTIONS ##########

	/**
  * Helper function that updates the button
  * state with an error or success class for
  * a specified duration
  * @param   {object}        $target         jQuery selection of the target button
  * @param   {string}        state           The state string that gets added to the loading class
  * @private
  */
	var _addButtonState = function _addButtonState($target, state) {
		var timer = setTimeout(function () {
			$target.removeClass(options.processingClass + ' ' + options.processingClass + state);
		}, options.processingDuration);

		$target.data('timer', timer).addClass(options.processingClass + state);
	};

	/**
  * Helper function to set the messages and the
  * button state.
  * @param       {object}    data                Result form the ajax request
  * @param       {object}    $form               jQuery selecion of the form
  * @param       {boolean}   disableButtons      If true, the button state gets set to (in)active
  * @param       {boolean}   showNoCombiMesssage If true, the error message for missing property combination selection will be displayed
  * @private
  */
	var _stateManager = function _stateManager(data, $form, disableButtons, showNoCombiSelectedMesssage) {

		// Remove the attribute images from the common content
		// so that it doesn't get rendered anymore. Then trigger
		// an event to the given selectors and deliver the
		// attrImages object
		if (options.attributImagesSwiper && data.attrImages && data.attrImages.length) {
			delete data.content.images;
			$(options.triggerAttrImagesTo).trigger(jse.libs.theme.events.SLIDES_UPDATE(), { attributes: data.attrImages });
		}

		// Set the messages given inside the data.content object
		$.each(data.content, function (i, v) {
			var $element = $body.hasClass('page-product-info') ? $this.find(options.selectorMapping[v.selector]) : $form.parent().find(options.selectorMapping[v.selector]);

			if ((!showNoCombiSelectedMesssage || v.value === '') && i === 'messageNoCombiSelected') {
				return true;
			}

			switch (v.type) {
				case 'hide':
					if (v.value === 'true') {
						$element.hide();
					} else {
						$element.show();
					}
					break;
				case 'html':
					$element.html(v.value);
					break;
				case 'attribute':
					$element.attr(v.key, v.value);
					break;
				case 'replace':
					if (v.value) {
						$element.replaceWith(v.value);
					} else {
						$element.addClass('hidden').empty();
					}
					break;
				default:
					$element.text(v.value);
					break;
			}
		});

		// Dis- / Enable the buttons
		if (disableButtons) {
			var $buttons = $form.find(options.cartButtons);
			if (data.success) {
				$buttons.removeClass('inactive');
				$buttons.removeClass('btn-inactive');
				$buttons.prop("disabled", false);
			} else {
				$buttons.addClass('inactive');
				$buttons.addClass('btn-inactive');
				$buttons.prop("disabled", true);
			}
		}

		if (data.content.message) {
			var $errorField = $form.find(options.selectorMapping[data.content.message.selector]);
			if (data.content.message.value) {
				$errorField.removeClass('hidden').show();
			} else {
				$errorField.addClass('hidden').hide();

				if (showNoCombiSelectedMesssage && data.content.messageNoCombiSelected !== undefined && data.content.messageNoCombiSelected) {
					if (data.content.messageNoCombiSelected.value) {
						$errorField.removeClass('hidden').show();
					} else {
						$errorField.addClass('hidden').hide();
					}
				}
			}
		}

		$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
	};

	/**
  * Helper function to send the ajax
  * On success redirect to a given url, open a layer with
  * a message or add the item to the cart-dropdown directly
  * (by triggering an event to the body)
  * @param       {object}      data      Form data
  * @param       {object}      $form     The form to fill
  * @param       {string}      url       The URL for the AJAX request
  * @private
  */
	var _addToSomewhere = function _addToSomewhere(data, $form, url, $button) {
		function callback() {
			jse.libs.xhr.post({ url: url, data: data }, true).done(function (result) {
				try {
					// Fill the page with the result from the ajax
					_stateManager(result, $form, false);

					// If the AJAX was successful execute
					// a custom functionality
					if (result.success) {
						switch (result.type) {
							case 'url':
								if (result.url.substr(0, 4) !== 'http') {
									location.href = jse.core.config.get('appUrl') + '/' + result.url;
								} else {
									location.href = result.url;
								}

								break;
							case 'dropdown':
								$body.trigger(jse.libs.theme.events.CART_UPDATE(), [true]);
								break;
							case 'layer':
								jse.libs.theme.modal.info({ title: result.title, content: result.msg });
								break;
							default:
								break;
						}
					}
				} catch (ignore) {}
				_addButtonState($button, '-success');
			}).fail(function () {
				_addButtonState($button, '-fail');
			}).always(function () {
				// Reset the busy flag to be able to perform
				// further AJAX requests
				busy = false;
			});
		}

		if (!busy) {
			// only execute the ajax
			// if there is no pending ajax call
			busy = true;

			jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.add, data, 500).then(callback).catch(callback);
		}
	};

	// ########## EVENT HANDLER ##########

	/**
  * Handler for the submit form / click
  * on "add to cart" & "wishlist" button.
  * It performs a check on the availability
  * of the combination and quantity. If
  * successful it performs the add to cart
  * or wishlist action, if it's not a
  * "check" call
  * @param       {object}    e      jQuery event object
  * @private
  */
	var _submitHandler = function _submitHandler(e) {
		if (e) {
			e.preventDefault();
		}

		var $self = $(this),
		    $form = $self.is('form') ? $self : $self.closest('form'),
		    customizer = $form.hasClass('customizer'),
		    properties = !!$form.find('.properties-selection-form').length,
		    module = properties ? '' : '/Attributes',
		    showNoCombiSelectedMesssage = e && e.data && e.data.target && e.data.target !== 'check';

		if ($form.length) {

			// Show properties overlay
			// to disable user interaction
			// before markup replace
			if (properties) {
				$this.addClass('loading');
			}

			if ($self.is('select')) {
				var price = $self.find(":selected").attr('data-price');
				$self.parents('.modifier-group').find('.selected-value-price').text(price);
			}

			var getGalleryHash = $('#current-gallery-hash').val();
			$form.find('#update-gallery-hash').val(getGalleryHash);

			var formdata = jse.libs.form.getData($form, null, true);
			formdata.target = e && e.data && e.data.target ? e.data.target : 'check';
			formdata.isProductInfo = $form.hasClass('product-info') ? 1 : 0;

			// Abort previous check ajax if
			// there is one in progress
			if (ajax && e) {
				ajax.abort();
			}

			// Add processing-class to the button
			// and remove old timed events
			if (formdata.target !== 'check') {
				var timer = $self.data('timer');
				if (timer) {
					clearTimeout(timer);
				}

				$self.removeClass(options.processingClass + '-success ' + options.processingClass + '-fail').addClass(options.processingClass);
			}

			ajax = jse.libs.xhr.get({
				url: options.checkUrl + module,
				data: formdata
			}, true).done(function (result) {
				_stateManager(result, $form, true, showNoCombiSelectedMesssage);
				$this.removeClass('loading');

				if (result.success) {
					var event = null,
					    url = null;

					switch (formdata.target) {
						case 'wishlist':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_WISHLIST();
							}
							url = options.wishlistUrl;
							break;
						case 'cart':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_CART();
								url = options.addCartCustomizerUrl;
							} else {
								url = options.addCartUrl;
							}
							break;
						case 'price_offer':
							$form.attr('action', options.priceOfferUrl).attr('method', options.priceOfferMethod);
							$form.off('submit');
							$form.submit();

							return;
						default:
							setTimeout(function () {
								$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
							}, 250);
							break;
					}

					if (event) {
						var deferred = $.Deferred();
						deferred.done(function (customizerRandom) {
							formdata[customizerRandom] = 0;
							_addToSomewhere(formdata, $form, url, $self);
						}).fail(function () {
							_addButtonState($self, '-fail');
						});
						$body.trigger(event, [{ 'deferred': deferred, 'dataset': formdata }]);
					} else if (url) {
						_addToSomewhere(formdata, $form, url, $self);
					}

					// Check if the gallery images changed
					if (formdata.target === 'check' && result.content.imageGallery.trim() !== '' && result.content.replaceGallery === true && formdata.isProductInfo === 1) {
						var loadingSpinner = jse.libs.loading_spinner.show($('.product-info-stage'), 9999);

						var swipers = [$('#product_image_swiper'), $('#product_thumbnail_swiper'), $('#product_thumbnail_swiper_mobile')];

						var _iteratorNormalCompletion = true;
						var _didIteratorError = false;
						var _iteratorError = undefined;

						try {
							for (var _iterator = swipers[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
								var element = _step.value;

								var instance = element.swiper();

								if ((typeof instance === 'undefined' ? 'undefined' : _typeof(instance)) !== 'object') {
									continue;
								}

								instance.destroy(true, true);
								element.off().remove();
							}
						} catch (err) {
							_didIteratorError = true;
							_iteratorError = err;
						} finally {
							try {
								if (!_iteratorNormalCompletion && _iterator.return) {
									_iterator.return();
								}
							} finally {
								if (_didIteratorError) {
									throw _iteratorError;
								}
							}
						}

						$('#image-collection-container').html(result.content.imageGallery);
						$('#product_image_layer').html(result.content.imageModal);

						gambio.widgets.init($('.product-info-content')).done(function () {
							jse.libs.loading_spinner.hide(loadingSpinner);
						});
					} else if (formdata.target === 'check' && result.content.imageGallery.trim() === '' && result.content.replaceGallery === true) {
						$('#image-collection-container').html(result.content.imageGallery);
						$('#product_image_layer').html(result.content.imageModal);
					}
				}
			}).fail(function () {
				_addButtonState($self, '-fail');
			});
		}
	};

	/**
  * Handler for the change property option
  * */
	var _changeProductOptions = function _changeProductOptions(e) {
		var option = e.currentTarget;
		var optionValue = $(option).data('value');
		var optionContainer = $(option).parents('.modifier-group');

		$(optionContainer).find('li.active').removeClass('active');
		$(optionContainer).find('.modifier-item.active-modifier').removeClass('active-modifier');
		$(optionContainer).find('input.hidden-input').val(optionValue);
		$(optionContainer).find('input.hidden-input').trigger('blur', []);

		$(option).parents('li').addClass('active');
		$(option).addClass('active-modifier');
	};

	var _selectSelectedModifierInfo = function _selectSelectedModifierInfo(e) {
		var option = e.currentTarget;
		var price = $(option).attr('data-price');
		var label = $(option).attr('data-label');
		$(option).parents('.modifier-group').find('.selected-value-price').removeClass('temporary-value').attr('data-default-price', price);
		$(option).parents('.modifier-group').find('.selected-value').attr('data-default-value', label);
	};

	var _setSelectedModifierInfo = function _setSelectedModifierInfo(e) {
		var option = e.currentTarget;
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var price = $(option).attr('data-price');
			var label = $(option).attr('data-label');
			$(option).parents('.modifier-group').find('.selected-value-price').addClass('temporary-value').text(price);
			$(option).parents('.modifier-group').find('.selected-value').text(label);
		}
	};

	var _resetSelectedModifierInfo = function _resetSelectedModifierInfo(e) {
		var option = $(this);
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var priceHolder = $(option).parents('.modifier-group').find('.selected-value-price');
			var labelHolder = $(option).parents('.modifier-group').find('.selected-value');
			$(priceHolder).removeClass('temporary-value').text($(priceHolder).attr('data-default-price'));
			$(labelHolder).text($(labelHolder).attr('data-default-value'));
		}
	};

	/**
  * Keyup handler for quantity input field
  *
  * @param e
  * @private
  */
	var _keyupHandler = function _keyupHandler(e) {
		clearTimeout(timeout);

		timeout = setTimeout(function () {
			_submitHandler.call(this, e);
		}.bind(this), 300);
	};

	/**
  * Event handler for the add to cart button, that shows or hides the throbber.
  */
	var _addToCartThrobberHandler = function _addToCartThrobberHandler(e) {
		var $btn = $(this);
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var formReady = true;

		$(".properties-selection-form select").each(function () {
			var val = $(this).val();
			if (!val || val < 1) {
				formReady = false;
			}
		});

		if (formReady) {
			$btn.hide();
			$btnFake.show().prop("disabled", true).prepend('<span class="throbbler"></span>');
		}
	};

	/**
  * Cart dropdown oben event handler for the body.
  */
	var _cartDropdownOpenHandler = function _cartDropdownOpenHandler(e) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var fakeOrigLabel = $btnFake.html();
		var productCount = $(".cart-products-count").html();

		var textPhrases = JSON.parse($('#product-details-text-phrases').html());
		console.log(textPhrases['productsInCartSuffix']);

		$btnFake.html("<i class=\"fa fa-check\"></i> " + parseInt(productCount) + textPhrases['productsInCartSuffix']).prop("disabled", true).addClass("btn-buy-complete");

		setTimeout(function () {
			$btnFake.html(fakeOrigLabel).removeClass("btn-buy-complete").hide().prop("disabled", false);
			$(".throbbler", $btn).remove();
			$btn.show();
		}, 5000);
	};

	// ########## INITIALIZATION ##########

	/**
  * Init function of the widget
  * @constructor
  */
	module.init = function (done) {

		var $forms = $this.find('form');

		if (options.page === 'product-info') {
			$forms.find("[name=btn-add-to-cart]").on('touchstart touchmove touchend touchcancel', function () {
				return $forms.find("[name=btn-add-to-cart]").click();
			});
			$forms.find("[name=btn-add-to-cart]").on('mouseup', _addToCartThrobberHandler);
			$("body").on('CART_DROPDOWN_OPEN', _cartDropdownOpenHandler);
		}

		$forms.on('submit', { 'target': 'cart' }, _submitHandler).on('click', options.wishlistButtons, { 'target': 'wishlist' }, _submitHandler).on('click', options.priceOfferButtons, { 'target': 'price_offer' }, _submitHandler).on('change', options.attributes, { 'target': 'check' }, _submitHandler).on('mouseover', options.attributes, _setSelectedModifierInfo).on('mouseout', options.attributes, _resetSelectedModifierInfo).on('blur', options.productOptionField, { 'target': 'check' }, _submitHandler).on('click', options.productOptions, { 'target': 'check' }, function (e) {
			_selectSelectedModifierInfo(e);
			_changeProductOptions(e);
		}).on('mouseover', options.productOptions, _setSelectedModifierInfo).on('mouseout', options.productOptions, _resetSelectedModifierInfo).on('blur', options.quantity, { 'target': 'check' }, function (e) {
			_submitHandler(e);
		}).on('keyup', options.quantity, { 'target': 'check' }, _keyupHandler);

		// Fallback if the backend renders incorrect data
		// on initial page call
		$forms.not('.no-status-check').not('.product-info').each(function () {
			_submitHandler.call($(this));
		});
		done();
	};

	// Return data to widget engine
	return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
